function [density, dz] = density_integrated_ajd5(x0, k, theta, sigma, L, mu, horizon, N, max_z)
% --------------------------------------------------------------------------------------------------
% Calculate the density of an integrated affine-jump-diffusion, i.e. Z=int_0^T(x_t dt), at point z,
% and for various time horizons (since simultaneous evaluation is fast). This is done using Fourier
% inversion of the transform of the integrated AJD.
% --------------------------------------------------------------------------------------------------
% x0                        ... initial intensity of AJD
% [k, theta, sigma, L, mu]  ... parameters of AJD dynamics
% horizon                   ... time horizon for which to compute the density
% N                         ... number of points for numerical integration of transfrom
% max_z                     ... optional: maximum value where transform is of interest
% --------------------------------------------------------------------------------------------------
% sample call: density_integrated_ajd5(0.001, 0.0022, 0.0656, 0.0415, 0.0180, 0.0607, 5, 1000)
%              density_integrated_ajd5(0.001, 0.0022, 0.0656, 0.0415, 0.0180, 0.0607, 5, 10000)
% --------------------------------------------------------------------------------------------------

% % Choose dz dynamically depending on parameters of AJD and horizon
% if (nargin <= 9)
%     T = max(horizon, 0.01);
%     tmp = (1-exp(-k*T))/k;
%     tmp2 = (1-exp(-2*k*T))/(2*k);
%     average = (max(0,theta)*T + x0*tmp) / T;           % average exptected value of intensity over horizon
%     max_integrated = average*T + ...                                             % from deterministic drift
%                      8*sigma*sqrt(abs(theta))/abs(k)*sqrt(T-2*tmp+tmp2) + ...    % five times std-dev. of integrated OU process
%                      gaminv(max(0.999, 1-1e-9/(L*T)), L*T, mu);                  % sum of jumps has gamma distribution
%     dz = max_integrated/(N-1);
% else
%     dz = max_z / N;
% end
% dz = dz/4;

% Pre-allocate memory
% dz_inv = 1 /dz / N * 2*pi;
%grid_z = zeros(1, N);
N_sparse = min(1024, N/2);
sparse_zero_grid = zeros(N_sparse, 1);

% Evaluate the transform on a grid of u-values
% C-Implemementation combined with spline interpolation, where max_z is chosen adaptively
if (1)
    % Evaluate Real(alpha)+Real(beta)*x0 (i.e. logarithm of absolute value of transform) of a sparse
    % grid to determine cut-off point for Fourier transform: amplitude < 10^-6
    grid_u = exp(0:20)';
    log_ft_real = log_transform2_integrated_ajd_c(zeros(size(grid_u)), grid_u, x0, k, theta, sigma, L, mu, horizon);
    
    % Determine cut-off point in frequency domain
    eps = log(5e-4);
    cut_off_pos = find(log_ft_real < eps, 1, 'first');
    if isempty(cut_off_pos)
        error('Fourier transform still not close to zero.');
    end
    if (cut_off_pos==1)
        max_dz_inv = grid_u(cut_off_pos);
    else
        % Linear interpolation to get exact cut-off
        a = log_ft_real(cut_off_pos-1);
        b = log_ft_real(cut_off_pos);
        max_dz_inv = grid_u(cut_off_pos) * (1 + (eps - a) / (b-a)); 
    end
    dz_inv = max_dz_inv / N;
    
    % Make sparse grid: 1024 points only
    tmp = log(N-1);
    tmp2 = tmp - log(101);
    used_grid = [];
    counter = 0;
    while (length(used_grid) < N_sparse)
        used_grid = [0:100 unique(round(exp(log(101):(tmp2/(N_sparse-100+counter)):tmp))) N-1];
        counter = counter + (N_sparse-length(used_grid)) * 1.2;
    end
    u_sparse = dz_inv * used_grid(1:N_sparse)';

    % Update dz, dz_inv
    dz_inv = u_sparse(end) / N;
    dz = 1 / dz_inv / N * 2 * pi;       
    
    if (1)
        % Calculate transform
        [t_real_sparse, t_imag_sparse] = transform2_integrated_ajd_c(sparse_zero_grid, u_sparse, x0, k, theta, sigma, L, mu, horizon);
        % plot(u_sparse,t_real_sparse)
        % plot(u_sparse,t_imag_sparse) 
        
        % Use spline interpolation
        if (1)
            % grid_inverse = (0:(dz_inv/100):((N-1)*dz_inv));
            t_real_interp = cubic_interpolation_matlab_c(u_sparse', t_real_sparse', dz_inv, N);
            t_imag_interp = cubic_interpolation_matlab_c(u_sparse', t_imag_sparse', dz_inv, N);
        else
            % Update grid_z_inv
            grid_inverse = (0:dz_inv:((N-1)*dz_inv));
    
            t_real_interp = interp1(u_sparse', t_real_sparse', grid_inverse, 'spline');
            t_imag_interp = interp1(u_sparse', t_imag_sparse', grid_inverse, 'spline');
        end
        transform = complex(t_real_interp, t_imag_interp);
    else
        % Evaluate real and imaginary part of characteristic function
        [log_char_real_sparse, log_char_imag_sparse] = log_transform2_integrated_ajd_c(sparse_zero_grid, u_sparse, x0, k, theta, sigma, L, mu, horizon); 
                        
        % Interpolate real and imaginary part of log-characteristic function
        if (0)
            log_char_real_interp = cubic_interpolation_matlab_c(u_sparse', log_char_real_sparse', dz_inv, N);
            log_char_imag_interp = cubic_interpolation_matlab_c(u_sparse', log_char_imag_sparse', dz_inv, N);

            % Calculate characteristic function
            transform = exp(complex(log_char_real_interp, log_char_imag_interp));
        else
            transform = get_interpolated_transform(u_sparse', log_char_real_sparse', log_char_imag_sparse', dz_inv, N);
        end
    end    
end

% Use FFT to calculate density
clear t_real_interp t_imag_interp;
tmp = real(transform(1));   % this term is counted twice in Fourier inversion integral
real_FFT = real(fft(transform));
clear transform;
integral_length = dz*N;
density_t = (real_FFT * (dz_inv/pi) - tmp/integral_length);
if (abs(sum(density_t)*dz - 1) > 0.1 )
    warning('Density of integrated common factor does not sum to one.')
end    

% Remove numerical artefacts from FFT
if (0)  % Matlab version
    % Part I (negative density)
    neg_pos = (density_t < 0);
    density_t(neg_pos) = 0;

    % Part II (force unimodality - right from peak)  
    [trash, pos] = max(density_t);
    cum_min = -cum_max_c(-density_t(pos:end));
    density_t(pos:end) = min(cum_min, density_t(pos:end));

    % Part III (force unimodality - left from peak)
    cum_min = -cum_max_c(-density_t(pos:(-1):1));
    cum_min = cum_min(pos:(-1):1);
    density_t(1:pos) = min(cum_min, density_t(1:pos));
    density = density_t / (sum(density_t) * dz);
else    % Fast version in C
    density = remove_artefacts_c(density_t, dz);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Optional post-processing %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Calculate expected value of integrated AJD
if(0)
    grid_z = 0:dz:((N-1)*dz);
    mean_density = [means sum(density_t .* grid_z)*dz];
    second_moment_density = sum(density_t .* grid_z.^2)*dz;
end

%disp(means*1e4);
%plot(horizons, means)

% Adjust density to match theoretical mean as implied by the first derivative of the characteristic
% function at zero
% eps = 1e-3;
% theoretical_means = zeros(1, num_horizons);
% for j=1:num_horizons
%     [trash, trans_eps_imag] = transform2_integrated_ajd_c(0, eps, x0, k, theta, sigma, L, mu, horizons(j));
%     theoretical_means(j) = trans_eps_imag / eps;
% end
% [trash, trans_2eps_imag] = transform2_integrated_ajd_c(0, 2*eps, x0, k, theta, sigma, L, mu, horizons);
% theoretical_2nd_moment = -(trans_2eps_imag - 2*trans_eps_imag) / eps^2;
% theoretical_std = sqrt(theoretical_2nd_moment - theoretical_mean^2)
%disp((theoretical_means - means)./theoretical_means);
if (0)
    % Correct by scaling dz
    correction_ratios = theoretical_means ./ means;
    dz = dz .* correction_ratios;
end
if (0)
    % Correct by scaling density directly
    alpha = (means - theoretical_means) ./ (theoretical_means .* means - second_moments);
    for j=1:num_horizons
        density(j,:) = density(j,:) .* (1 + alpha(j) .* grid_z(j,:));
        density(j,:) = density(j,:) ./ (sum(density(j,:)) * dz(j));
    end
end




